<?php
// Enable error reporting (disable in production)
ini_set('display_errors', 0);
error_reporting(E_ALL);

require 'vendor/autoload.php';

use Google\Cloud\Vision\V1\ImageAnnotatorClient;
require 'config.php';

function log_message($message) {
    file_put_contents(__DIR__ . '/webhook_log.txt', date("Y-m-d H:i:s") . " - " . $message . "\n", FILE_APPEND);
}

function handle_image_upload($file, $baseFilename, $suffix) {
    $imagesDir = __DIR__ . '/img';
    $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/gif'];
    if (!in_array($file['type'], $allowedMimeTypes)) {
        log_message("Unsupported image type for {$suffix}: " . ($file['type'] ?? 'unknown'));
        return null;
    }
    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $newName = "{$baseFilename}_{$suffix}.{$ext}";
    $targetPath = "{$imagesDir}/{$newName}";
    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
        log_message("Saved {$suffix} image to {$targetPath}");
        return "img/{$newName}";
    }
    log_message("Failed to move uploaded file for {$suffix} to {$targetPath}");
    return null;
}

putenv('GOOGLE_APPLICATION_CREDENTIALS=/home/brafasystems/credentials/hip-watch-445016-b7-7d7816efb7af.json');

$credentialsPath = '/home/brafasystems/credentials/hip-watch-445016-b7-7d7816efb7af.json';
if (!file_exists($credentialsPath) || !is_readable($credentialsPath)) {
    log_message("Credentials file missing or not readable.");
    http_response_code(500);
    exit("Server configuration error.");
}

$imagesDir = __DIR__ . '/img';
if (!is_dir($imagesDir)) mkdir($imagesDir, 0755, true);
if (!is_writable($imagesDir)) {
    log_message("Images directory not writable.");
    http_response_code(500);
    exit("Server configuration error.");
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit("Method not allowed.");
}

$anprXmlFile = $_FILES['anpr_xml']['tmp_name'] ?? null;
$detectionImage = $_FILES['detectionPicture_jpg']['tmp_name'] ?? null;
$licensePlateImage = $_FILES['licensePlatePicture_jpg']['tmp_name'] ?? null;
$vehicleImage = $_FILES['vehiclePicture_jpg']['tmp_name'] ?? null;
$additionalImage3 = $_FILES['additionalImage3_jpg']['tmp_name'] ?? null;

if (!$anprXmlFile || !$detectionImage) {
    http_response_code(400);
    exit("Missing required files.");
}

$xmlContent = file_get_contents($anprXmlFile);
$xml = simplexml_load_string($xmlContent, 'SimpleXMLElement', LIBXML_NOCDATA);
if (!$xml) {
    http_response_code(400);
    exit("Invalid XML format.");
}

$namespaces = $xml->getNamespaces(true);
$defaultNS = $namespaces[''] ?? 'http://www.hikvision.com/ver20/XMLSchema';
$xml->registerXPathNamespace('h', $defaultNS);
$xml->registerXPathNamespace('ns', $defaultNS);

$ipNodes = $xml->xpath('/h:EventNotificationAlert/h:ipAddress');
$ipAddress = !empty($ipNodes) ? trim((string)$ipNodes[0]) : ($_SERVER['REMOTE_ADDR'] ?? '');

$directionNodes = $xml->xpath('//ns:ANPR/ns:direction');
$xmlDirection = !empty($directionNodes) ? trim((string)$directionNodes[0]) : '';
$outCameraIPs = ['192.168.71.165'];
$isOutCamera = in_array($ipAddress, $outCameraIPs) || strtolower($xmlDirection) === 'forward';
log_message("Determined isOutCamera: " . ($isOutCamera ? 'true' : 'false'));

$licensePlateNodes = $xml->xpath('//ns:ANPR/ns:licensePlate');
$dateTimeNodes = $xml->xpath('//ns:dateTime');
if (empty($dateTimeNodes)) {
    http_response_code(400);
    exit("Missing required field: dateTime.");
}
$licensePlate = !empty($licensePlateNodes) ? trim((string)$licensePlateNodes[0]) : 'unknown';
$dateTime = trim((string)$dateTimeNodes[0]);
$mysqlDateTime = date("Y-m-d H:i:s", strtotime($dateTime));
if (strtolower($licensePlate) === 'unknown') $licensePlate = 'unknown';

$sanitizedLicensePlate = preg_replace('/[^A-Za-z0-9\-]/', '_', $licensePlate);
$timestamp = date("YmdHis", strtotime($dateTime));
$directionSuffix = $isOutCamera ? 'out' : 'in';
$baseFilename = "{$sanitizedLicensePlate}_{$timestamp}_{$directionSuffix}";

$savedXmlPath = "img/{$baseFilename}.xml";
if (move_uploaded_file($anprXmlFile, __DIR__ . "/{$savedXmlPath}")) {
    log_message("Saved XML to " . __DIR__ . "/{$savedXmlPath}");
} else {
    log_message('Failed to save XML file');
}
$savedDetectionImagePath = handle_image_upload($_FILES['detectionPicture_jpg'], $baseFilename, 'detection');
if (!$savedDetectionImagePath) {
    log_message('Failed to save required detection image');
    http_response_code(500);
    exit('Failed to save detection image.');
}
$savedLicensePlateImagePath = $licensePlateImage ? handle_image_upload($_FILES['licensePlatePicture_jpg'], $baseFilename, 'licensePlate') : ($vehicleImage ? handle_image_upload($_FILES['vehiclePicture_jpg'], $baseFilename, 'vehicle') : null);
$savedImage3Path = isset($_FILES['additionalImage3_jpg']) ? handle_image_upload($_FILES['additionalImage3_jpg'], $baseFilename, 'additional3') : null;

$convertPath = '/usr/bin/convert';
if (!file_exists($convertPath)) {
    log_message("ImageMagick 'convert' not found.");
    http_response_code(500);
    exit("Server configuration error.");
}

$weightToInsert = 2000;
$image3 = null;
if ($isOutCamera) {
    $outCropBox = "178x156+1408+74";
    $croppedOutFilename = "{$baseFilename}_weight_out.jpg";
    $croppedOutPath = __DIR__ . "/img/{$croppedOutFilename}";
    $outOcrOutputPath = __DIR__ . "/img/{$baseFilename}_weight_out_ocr_output.txt";
    
    $cropCommand = "$convertPath " . escapeshellarg(__DIR__ . "/{$savedDetectionImagePath}") . " -crop $outCropBox +repage " . escapeshellarg($croppedOutPath);
    shell_exec($cropCommand);

    if (file_exists($croppedOutPath)) {
        try {
            $annotator = new ImageAnnotatorClient(['credentials' => $credentialsPath]);
            $response = $annotator->textDetection(file_get_contents($croppedOutPath));
            $annotator->close();
            $texts = $response->getTextAnnotations();
            $ocrOutput = isset($texts[0]) ? trim($texts[0]->getDescription()) : '';
            file_put_contents($outOcrOutputPath, $ocrOutput);
            if (preg_match('/\d+/', preg_replace('/\s+/', '', $ocrOutput), $matches)) {
                $weightToInsert = (int)$matches[0];
            }
        } catch (Exception $e) {
            log_message("OCR Exception: " . $e->getMessage());
        }
    }
    $image3 = file_exists($croppedOutPath) ? "img/{$croppedOutFilename}" : ($savedImage3Path ?? null);
    $direction = 'out';
    $imageColumn1 = 'out_image';
    $imageColumn2 = 'out_image2';
    $imageColumn3 = 'out_image3';
} else {
    $cropBox = "161x89+658+17";
    $croppedImageFilename = "{$baseFilename}_weight_in.jpg";
    $croppedImagePath = __DIR__ . "/img/{$croppedImageFilename}";
    $ocrOutputPath = __DIR__ . "/img/{$baseFilename}_weight_ocr_output.txt";

    $cropCommand = "$convertPath " . escapeshellarg(__DIR__ . "/{$savedDetectionImagePath}") . " -crop $cropBox +repage " . escapeshellarg($croppedImagePath);
    shell_exec($cropCommand);

    if (file_exists($croppedImagePath)) {
        try {
            $annotator = new ImageAnnotatorClient(['credentials' => $credentialsPath]);
            $response = $annotator->textDetection(file_get_contents($croppedImagePath));
            $annotator->close();
            $texts = $response->getTextAnnotations();
            $ocrOutput = isset($texts[0]) ? trim($texts[0]->getDescription()) : '';
            file_put_contents($ocrOutputPath, $ocrOutput);
            if (preg_match('/\d+/', preg_replace('/\s+/', '', $ocrOutput), $matches)) {
                $weightToInsert = (int)$matches[0];
            }
        } catch (Exception $e) {
            log_message("OCR Exception: " . $e->getMessage());
        }
    }
    $image3 = $savedImage3Path ?? "img/{$croppedImageFilename}";
    $direction = 'in';
    $imageColumn1 = 'image1';
    $imageColumn2 = 'image2';
    $imageColumn3 = 'image3';
}

try {
    $pdo->beginTransaction();
    $sql = "INSERT INTO entries (license_plate, direction, timestamp, weight_kg, processed, {$imageColumn1}, {$imageColumn2}, {$imageColumn3}) VALUES (:license_plate, :direction, :timestamp, :weight_kg, 0, :image1, :image2, :image3)";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':license_plate' => $licensePlate,
        ':direction' => $direction,
        ':timestamp' => $mysqlDateTime,
        ':weight_kg' => $weightToInsert,
        ':image1' => $savedDetectionImagePath,
        ':image2' => $savedLicensePlateImagePath,
        ':image3' => $image3
    ]);
    $pdo->commit();
    $insertId = $pdo->lastInsertId();
    log_message(
        "Inserted '{$direction}' entry ID={$insertId} with images: " .
        "{$imageColumn1}={$savedDetectionImagePath}, " .
        "{$imageColumn2}={$savedLicensePlateImagePath}, " .
        "{$imageColumn3}={$image3}"
    );
    http_response_code(200);
    echo "Webhook data processed and stored successfully.";
} catch (PDOException $e) {
    $pdo->rollBack();
    log_message("DB Insert Error: " . $e->getMessage());
    http_response_code(500);
    echo "Database error: " . $e->getMessage();
}
exit;
?>
