<?php
// update_plate.php
session_start();
require 'config.php';

// Ensure the user is authenticated
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized access.']);
    exit;
}

// Input validation
$in_id = $_POST['in_id'] ?? null;
$out_id = $_POST['out_id'] ?? null;
$new_plate = $_POST['new_plate'] ?? null;

if (!$in_id || !$out_id || !$new_plate) {
    echo json_encode(['status' => 'error', 'message' => 'Missing required data.']);
    exit;
}

// Optionally, sanitize the new_plate to prevent filename issues
$sanitized_new_plate = preg_replace('/[^A-Za-z0-9\-]/', '_', $new_plate);

// Fetch entries
$stmt = $pdo->prepare("SELECT * FROM entries WHERE id IN (:in_id, :out_id)");
$stmt->execute([':in_id' => $in_id, ':out_id' => $out_id]);
$entries = $stmt->fetchAll(PDO::FETCH_ASSOC);

if (count($entries) < 2) {
    echo json_encode(['status' => 'error', 'message' => 'Could not find matching in/out records.']);
    exit;
}

// Identify in/out entries
$in_entry = null;
$out_entry = null;
foreach ($entries as $e) {
    if ($e['direction'] === 'in') $in_entry = $e;
    if ($e['direction'] === 'out') $out_entry = $e;
}

// Both must be present
if (!$in_entry || !$out_entry) {
    echo json_encode(['status' => 'error', 'message' => 'In/out pair not found.']);
    exit;
}

// Extract timestamp from in_entry for filename consistency
$timestamp = date("YmdHis", strtotime($in_entry['timestamp']));

// Construct old base filename (from old plate)
$old_plate = $in_entry['license_plate'];
$old_base = preg_replace('/[^A-Za-z0-9\-]/', '_', $old_plate) . "_{$timestamp}";
if ($old_plate === 'unknown') {
    $old_base = "unknown_{$timestamp}";
}

// Construct new base filename
$new_base = $sanitized_new_plate . "_{$timestamp}";

// Function to rename files safely
function rename_file_if_exists($oldPath, $newPath) {
    if (file_exists($oldPath)) {
        if (!rename($oldPath, $newPath)) {
            return false;
        }
    }
    return true;
}

// Prepare arrays of old->new filenames
$updateFields = [];
$imageFields = ['image1', 'image2', 'image3', 'out_image', 'out_image2', 'out_image3'];

// Loop through image fields to rename files
foreach ($imageFields as $field) {
    // For 'in' entry
    if (!empty($in_entry[$field])) {
        $oldValueIn = $in_entry[$field];
        if (strpos($oldValueIn, $old_base) !== false) {
            $oldFullPath = __DIR__ . "/" . $oldValueIn;
            $ext = pathinfo($oldFullPath, PATHINFO_EXTENSION);
            // Determine suffix from old filename
            preg_match('/_(detection|licensePlate|vehicle|additional3|weight_in)\./i', $oldValueIn, $matches);
            $suffix = $matches[1] ?? 'detection';
            $newFileName = "{$new_base}_{$suffix}.{$ext}";
            $newFullPath = __DIR__ . "/img/" . $newFileName;

            if (!rename_file_if_exists($oldFullPath, $newFullPath)) {
                echo json_encode(['status' => 'error', 'message' => "Could not rename {$oldFullPath} to {$newFullPath}"]);
                exit;
            }
            $updateFields['in'][$field] = "img/$newFileName";
        }
    }

    // For 'out' entry
    if (!empty($out_entry[$field])) {
        $oldValueOut = $out_entry[$field];
        if (strpos($oldValueOut, $old_base) !== false) {
            $oldFullPath = __DIR__ . "/" . $oldValueOut;
            $ext = pathinfo($oldFullPath, PATHINFO_EXTENSION);
            // Determine suffix for out image from filename. Older versions
            // sometimes produced filenames like "*_out_out.jpg". Detect both
            // "*_out.jpg" and "*_out_out.jpg" and rename to the single "_out" format.
            preg_match('/_out(?:_out)?\./i', $oldValueOut, $matches);
            $suffix = 'out';
            $newFileName = "{$new_base}_{$suffix}.{$ext}";
            $newFullPath = __DIR__ . "/img/" . $newFileName;

            if (!rename_file_if_exists($oldFullPath, $newFullPath)) {
                echo json_encode(['status' => 'error', 'message' => "Could not rename {$oldFullPath} to {$newFullPath}"]);
                exit;
            }
            $updateFields['out'][$field] = "img/$newFileName";
        }
    }
}

// Rename XML file as well
$oldXml = __DIR__ . "/img/{$old_base}.xml";
$newXml = __DIR__ . "/img/{$new_base}.xml";
if (file_exists($oldXml)) {
    if (!rename_file_if_exists($oldXml, $newXml)) {
        echo json_encode(['status' => 'error', 'message' => "Could not rename XML file from {$oldXml} to {$newXml}"]);
        exit;
    }
}

// Update the database with the new plate and filenames
try {
    $pdo->beginTransaction();

    // Update IN entry
    $inUpdateFieldsSql = [];
    $inParams = [':new_plate' => $new_plate, ':in_id' => $in_id];
    foreach ($updateFields['in'] ?? [] as $f => $val) {
        $inUpdateFieldsSql[] = "$f = :$f";
        $inParams[":$f"] = $val;
    }
    if (!empty($inUpdateFieldsSql)) {
        $inSql = "UPDATE entries SET license_plate = :new_plate, " . implode(", ", $inUpdateFieldsSql) . " WHERE id = :in_id";
    } else {
        $inSql = "UPDATE entries SET license_plate = :new_plate WHERE id = :in_id";
    }
    $stmt = $pdo->prepare($inSql);
    $stmt->execute($inParams);

    // Update OUT entry
    $outUpdateFieldsSql = [];
    $outParams = [':new_plate' => $new_plate, ':out_id' => $out_id];
    foreach ($updateFields['out'] ?? [] as $f => $val) {
        $outUpdateFieldsSql[] = "$f = :$f";
        $outParams[":$f"] = $val;
    }
    if (!empty($outUpdateFieldsSql)) {
        $outSql = "UPDATE entries SET license_plate = :new_plate, " . implode(", ", $outUpdateFieldsSql) . " WHERE id = :out_id";
    } else {
        $outSql = "UPDATE entries SET license_plate = :new_plate WHERE id = :out_id";
    }
    $stmt = $pdo->prepare($outSql);
    $stmt->execute($outParams);

    $pdo->commit();

    echo json_encode(['status' => 'success', 'message' => 'Plate updated successfully.']);
} catch (PDOException $e) {
    $pdo->rollBack();
    echo json_encode(['status' => 'error', 'message' => 'Database error: ' . $e->getMessage()]);
}
?>
