<?php
// export_transactions.php
session_start();
require 'config.php';

// Redirect to login if not authenticated
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// Fetch filter parameters
$search = $_GET['search'] ?? '';
$rubbish_type_id = $_GET['rubbish_type_id'] ?? '';
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';

// Prepare Transactions Query with Filters
$transactionsSql = "SELECT 
                        p.id as payment_id,
                        p.license_plate, 
                        p.weight_diff, 
                        p.total_amount, 
                        p.timestamp as payment_time,
                        r.name as rubbish_type,
                        r.rate_per_kg
                    FROM payments p
                    JOIN rubbish_types r ON p.rubbish_type_id = r.id
                    WHERE p.license_plate LIKE :search";

$bindParams = [
    ':search' => '%' . $search . '%'
];

if (!empty($rubbish_type_id)) {
    $transactionsSql .= " AND p.rubbish_type_id = :rubbish_type_id";
    $bindParams[':rubbish_type_id'] = $rubbish_type_id;
}

if (!empty($start_date)) {
    $transactionsSql .= " AND p.timestamp >= :start_date";
    $bindParams[':start_date'] = $start_date . ' 00:00:00';
}

if (!empty($end_date)) {
    $transactionsSql .= " AND p.timestamp <= :end_date";
    $bindParams[':end_date'] = $end_date . ' 23:59:59';
}

$transactionsSql .= " ORDER BY p.timestamp DESC";
$transactionsStmt = $pdo->prepare($transactionsSql);

// Bind parameters
foreach ($bindParams as $key => $value) {
    if ($key === ':rubbish_type_id') {
        $transactionsStmt->bindValue($key, $value, PDO::PARAM_INT);
    } else {
        $transactionsStmt->bindValue($key, $value);
    }
}

$transactionsStmt->execute();
$transactionRows = $transactionsStmt->fetchAll(PDO::FETCH_ASSOC);

// Set headers for UTF-8 and CSV
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename=transactions_' . date('Y-m-d') . '.csv');

// Output BOM for UTF-8
echo "\xEF\xBB\xBF";

// Open output stream
$output = fopen('php://output', 'w');

// Define the CSV headers
fputcsv($output, ['Payment ID', 'License Plate', 'Rubbish Type', 'Weight Difference (kg)', 'Rate per kg', 'Total Amount (kr)', 'Payment Time']);

// Write transaction data
foreach ($transactionRows as $trx) {
    fputcsv($output, [
        $trx['payment_id'],
        $trx['license_plate'],
        $trx['rubbish_type'],
        $trx['weight_diff'],
        number_format($trx['rate_per_kg'], 2, '.', ''), // Ensure decimal format
        number_format($trx['total_amount'], 2, '.', ''),
        $trx['payment_time']
    ]);
}

fclose($output);
exit;
?>
