<?php
require 'config.php';

// Retrieve form data
$in_id = $_POST['in_id'] ?? null;
$out_id = $_POST['out_id'] ?? null;
$weight_diff = $_POST['weight_diff'] ?? null;
$rubbish_type_id = $_POST['rubbish_type_id'] ?? null;

// Check if the plate is manual or existing
$manual_plate = $_POST['manual_plate'] ?? null;

if (!$in_id || !$out_id || !$weight_diff || !$rubbish_type_id) {
    // Handle missing data
    header("Location: interface.php?msg=Missing required data.");
    exit;
}

// Determine the license plate
if ($manual_plate && !empty(trim($manual_plate))) {
    // Validate the manual_plate format
    $manual_plate = strtoupper(trim($manual_plate));
    if (!preg_match('/^[A-Z]{2}[A-Z0-9]{1}[0-9]{2}$/', $manual_plate)) {
        header("Location: interface.php?msg=Invalid manual license plate format.");
        exit;
    }
    $license_plate = $manual_plate;
} else {
    // Fetch the existing license plate from the 'in' entry
    $stmt = $pdo->prepare("SELECT license_plate FROM entries WHERE id = :in_id AND direction = 'in'");
    $stmt->execute([':in_id' => $in_id]);
    $entry = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($entry) {
        $license_plate = $entry['license_plate'];
    } else {
        header("Location: interface.php?msg=Invalid IN entry.");
        exit;
    }
}

// Fetch the rate per kg for the selected rubbish type
$stmt = $pdo->prepare("SELECT rate_per_kg FROM rubbish_types WHERE id = :type_id");
$stmt->execute([':type_id' => $rubbish_type_id]);
$type = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$type) {
    header("Location: interface.php?msg=Invalid rubbish type.");
    exit;
}
$rate_per_kg = (float)$type['rate_per_kg'];

// Calculate the total amount
$total_amount = $weight_diff * $rate_per_kg;

// Insert or update the 'in' and 'out' entries as processed
$pdo->beginTransaction();

try {
    // Update the 'in' entry
    $stmt = $pdo->prepare("UPDATE entries SET processed = 1 WHERE id = :in_id");
    $stmt->execute([':in_id' => $in_id]);

    // Update the 'out' entry
    $stmt = $pdo->prepare("UPDATE entries SET processed = 1 WHERE id = :out_id");
    $stmt->execute([':out_id' => $out_id]);

    // Insert a new payment record (assuming you have a payments table)
    $stmt = $pdo->prepare("INSERT INTO payments (license_plate, in_id, out_id, weight_diff, rubbish_type_id, total_amount, timestamp) VALUES (:license_plate, :in_id, :out_id, :weight_diff, :rubbish_type_id, :total_amount, NOW())");
    $stmt->execute([
        ':license_plate' => $license_plate,
        ':in_id' => $in_id,
        ':out_id' => $out_id,
        ':weight_diff' => $weight_diff,
        ':rubbish_type_id' => $rubbish_type_id,
        ':total_amount' => $total_amount
    ]);

    // Capture the payment ID for POS transaction logging
    $paymentId = $pdo->lastInsertId();

    $pdo->commit();

    // --- Send payment information to POS ---
    $posPayload = [
        "MessageHeader" => [
            "MessageClass" => "SERVICE",
            "MessageCategory" => "PAYMENT",
            "MessageType" => "REQUEST",
            "ServiceID" => "Service_" . uniqid(),
            "SaleID" => "SALE_" . uniqid(),
            "POIID" => "Brafa_T650P"
        ],
        "PaymentRequest" => [
            "SaleData" => [
                "SaleTransactionID" => [
                    "TransactionID" => uniqid("TX_"),
                    "TimeStamp" => gmdate('c')
                ]
            ],
            "PaymentTransaction" => [
                "AmountsReq" => [
                    "Currency" => "ISK",
                    "RequestedAmount" => (string)round($total_amount, 2)
                ]
            ]
        ]
    ];

    $ch = curl_init('https://cstpos.test-gsc.vfims.com/oidc/poscloud/nexo/v2/payment');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'x-site-entity-id: 0c9f1700-5638-46c7-bcfd-6e8a6af130dc',
        'x-terminal-simulator: false',
        'Content-Type: application/json',
        'Accept: application/json',
        'Authorization: Basic ZmIwMTBkZTctZjE5Zi00YTE0LWEyMWQtYjllMDZhMTEwNTZjOnZhcEFvZnVTbEp6b2VvTWpNaWdidXBibWF2bGVuWHhUckpwbA=='
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($posPayload));
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    $posResponse = curl_exec($ch);
    $posErr = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $logMessage = '[' . date('Y-m-d H:i:s') . " Payment Request HTTP={$httpCode} | " . json_encode($posPayload);
    if ($posErr) {
        $logMessage .= " | cURL Error: {$posErr}";
    }
    if ($posResponse !== false) {
        $logMessage .= " | POS Response: {$posResponse}";
    }
    file_put_contents(__DIR__ . '/pos_log.txt', $logMessage . "\n", FILE_APPEND);

    // Store request and response in pos_transactions table
    try {
        $stmt = $pdo->prepare("INSERT INTO pos_transactions (payment_id, request_json, response_json, http_status) VALUES (:payment_id, :request_json, :response_json, :http_status)");
        $stmt->execute([
            ':payment_id' => $paymentId,
            ':request_json' => json_encode($posPayload),
            ':response_json' => $posResponse !== false ? $posResponse : '',
            ':http_status' => $httpCode
        ]);
    } catch (PDOException $e) {
        file_put_contents(__DIR__ . '/webhook_log.txt', date('Y-m-d H:i:s') . ' - DB Error saving POS transaction: ' . $e->getMessage() . "\n", FILE_APPEND);
    }

    if ($posErr || $httpCode >= 300) {
        header("Location: interface.php?msg=POS payment failed.");
        exit;
    }

    header("Location: interface.php?msg=Payment processed successfully.");
    exit;
} catch (PDOException $e) {
    $pdo->rollBack();
    // Log the error
    file_put_contents(__DIR__ . '/webhook_log.txt', date("Y-m-d H:i:s") . " - DB Error in pay.php: " . $e->getMessage() . "\n", FILE_APPEND);
    header("Location: interface.php?msg=Database error.");
    exit;
}
?>